#include "lzw.h"
#include <windows.h>


#define LZW_MIN_BITS       9
#define LZW_MAX_BITS      12
#define LZW_CLEAR_CODE    (1 << (LZW_MIN_BITS - 1))
#define LZW_EOI_CODE      (LZW_CLEAR_CODE + 1)
#define LZW_TABLE_SIZE    (1 << LZW_MAX_BITS)
#define LZW_OUTBUF_SIZE   512

typedef BOOL (*LZWCALLBACK)( void* User, UCHAR* Buffer, DWORD BufferSize );
/*
  this function is called from lzw compressor to store output data
  BufferSize will always be multiple of 4 except the final call
*/

typedef struct TagLZWCTX {
  ULONG   LZW_TablePrefix[ LZW_TABLE_SIZE - LZW_EOI_CODE ];
  UCHAR   LZW_TableSuffix[ LZW_TABLE_SIZE - LZW_EOI_CODE ];
  ULONG   LZW_CodeSize, LZW_TableSize;

  // the following tables are used to speedup string search;
  // first, these tables are initialized to -1
  // when new element is inserted, they are updated

  int     LZW_NextSuffix[ LZW_TABLE_SIZE - LZW_EOI_CODE ];
  int     LZW_SuffixGroup[ 256 ];

  UCHAR  OutBuf[ 512 ];
  ULONG  OutBitIndex;
  LZWCALLBACK  Store;
  void*  User;
} LZWCTX;

static BOOL LZW_WriteCode( LZWCTX* Ctx, ULONG Code )
  {
    ULONG  Bit, Byte, i, b, cs;

    b = Ctx->OutBitIndex;
    cs = Ctx->LZW_CodeSize;
    Bit = b & 7;
    Byte = b >> 3;
    if( b + cs > LZW_OUTBUF_SIZE * 8 ) {
      i = Byte; Byte &= 3; i -= Byte;
      if( ! Ctx->Store( Ctx->User, Ctx->OutBuf, Byte ) ) return FALSE;
      OutBuf[0] = OutBuf[ Byte ];
      OutBuf[1] = OutBuf[ Byte + 1 ];
      OutBuf[2] = OutBuf[ Byte + 2 ];
      OutBuf[3] = OutBuf[ Byte + 3 ];
      b = Bit + (i << 3);
      Byte = i;
    }
    i = cs + Bit - 8;
    OutBuf[ Byte ] += (UCHAR) (Code >> i);
    if( i > 8 ) {
      OutBuf[ Byte + 1 ] = (UCHAR) (Code >> (i - 8));
      OutBuf[ Byte + 2 ] = (UCHAR) (Code << (16 - i));
    }
    else if( i == 8 )
      OutBuf[ Byte + 1 ] = (UCHAR) Code;
    else
      OutBuf[ Byte + 1 ] = (UCHAR) (Code << (8 - i));
    Ctx->OutBitIndex = b + cs;
    return TRUE;
  }

static BOOL LZW_IsStringInTable( LZWCTX* Ctx, ULONG* Prefix, UCHAR Suffix )
  {
    ULONG  p;
    int    next;

    p = *Prefix;
    if( p == LZW_CLEAR_CODE ) {
      *Prefix = Suffix;
      return TRUE;
    }
    // find Suffix in the table and if prefix is equal to *prefix,
    // string is in the table
    next = Ctx->LZW_SuffixGroup[ Suffix ];
    while( next != -1 ) {
      if( Ctx->LZW_TableSuffix[ next ] == Suffix &&
          Ctx-> LZW_TablePrefix[ next ] == p ) {
        *Prefix = next + LZW_EOI_CODE + 1;
        return TRUE;
      }
      next = Ctx->LZW_NextSuffix[ next ];
    }
    return FALSE;
  }

static void LZW_InitTables( LZWCTX* Ctx )
  {
    int  i;

    Ctx->LZW_CodeSize = LZW_MIN_BITS;
    Ctx->LZW_TableSize = LZW_EOI_CODE + 1;
    for( i = 0; i < 256; i++ ) Ctx->LZW_SuffixGroup[ i ] = -1;
    for( i = 0; i < LZW_TABLE_SIZE - LZW_EOI_CODE; i++ )
      Ctx->LZW_NextSuffix[ i ] = -1;
  }

static BOOL LZW_Compressor( LZWCTX* Ctx, UCHAR* InBuf, ULONG InBufSz,
                            LZWCALLBACK Callback, void* User )
// returns false if callback return false
  {
    UCHAR  Ch;
    ULONG  In;       // index
    ULONG  Str;      // current string is represented as index in the table
    int    i, j;

    LZW_InitTables( Ctx );
    Ctx->Store = Callback;
    Ctx->User = User;
    In = OutBitIndex = 0;
    Ctx->OutBuf[0] = 0; // we will not have to perform masking in WriteCode
    Str = LZW_CLEAR_CODE; // undefined value
    if( ! LZW_WriteCode( Ctx, LZW_CLEAR_CODE ) ) return FALSE;
    while( In < InBufSz ) {
      Ch = InBuf[ In++ ];
      if( ! LZW_IsStringInTable( Ctx, &Str, Ch ) ) {
        // add Str + new symbol to the table
        if( ! LZW_WriteCode( Ctx, Str ) ) return FALSE;
        if( Ctx->LZW_TableSize >= LZW_TABLE_SIZE - 2 ) { // bump code size
          if( ! LZW_WriteCode( Ctx, LZW_CLEAR_CODE ) ) return FALSE;
          LZW_InitTables( Ctx );
        }
        else {
          i = Ctx->LZW_TableSize - LZW_EOI_CODE - 1;
          Ctx->LZW_TablePrefix[ i ] = Str;
          Ctx->LZW_TableSuffix[ i ] = Ch;
          j = Ctx->LZW_SuffixGroup[ Ch ];
          Ctx->LZW_SuffixGroup[ Ch ] = i;
          Ctx->LZW_NextSuffix[ i ] = j;
          if( ++Ctx->LZW_TableSize > (1 << Ctx->LZW_CodeSize) - 1 )
            Ctx->LZW_CodeSize++;
        }
        Str = Ch;
      }
    }
    if( ! LZW_WriteCode( Ctx, Str ) ) return FALSE;
    if( ! LZW_WriteCode( Ctx, LZW_EOI_CODE ) ) return FALSE;
    if( Ctx->OutBitIndex == 0 ) return TRUE;
    return Callback( User, Ctx->OutBuf, (Ctx->OutBitIndex + 7) >> 3 );
  }
